<?php

require_once 'types.php';

class MOV extends Opcode {
    private Components $components;
    private Registers $registers;
    private Register $programCounter;
    private int $depth;

    public function __construct(
        Components &$components,
        Registers &$registers,
        int &$depth,
        Register &$programCounter
    ) {
        $this->components = $components;
        $this->registers = $registers;
        $this->programCounter = $programCounter;
        $this->depth = $depth;
    }

    public function execute(int $args) : void {
        $address = 0;
        for($offset = 1; $offset <= $this->depth; $offset++) {
            $address += $this->components->read($this->programCounter->read() + $offset) << (8 * ($this->depth - $offset));
        }
        $this->programCounter->write($this->programCounter->read() + $this->depth);
        if(($args & 0b100000) === 0b100000) { // if true register -> address, false address -> register
            $this->components->write($address, $this->registers->read($args & 0b111) & 0b11111111);
        } else {
            $this->registers->write($args & 0b111, $this->components->read($address));
        }
    }
}

class NAND extends Opcode {
    private Components $components;
    private Registers $registers;
    private Register $programCounter;
    private int $depth;

    private function nand(int $a, int $b) : int {
        return ~($a & $b);
    }

    public function __construct(
        Components &$components,
        Registers &$registers,
        int &$depth,
        Register &$programCounter
    ) {
        $this->components = $components;
        $this->registers = $registers;
        $this->programCounter = $programCounter;
        $this->depth = $depth;
    }

    public function execute(int $args) : void {
        $num1 = $this->registers->read(($args & 0b111000) >> 3);
        $num2 = $this->registers->read($args & 0b111);
        $this->registers->write(($args & 0b111000) >> 3, $this->nand($num1, $num2));
    }
}


class BIT extends Opcode {
    private Components $components;
    private Registers $registers;
    private Register $programCounter;
    private int $depth;

    public function __construct(
        Components &$components,
        Registers &$registers,
        int &$depth,
        Register &$programCounter
    ) {
        $this->components = $components;
        $this->registers = $registers;
        $this->programCounter = $programCounter;
        $this->depth = $depth;
    }

    public function execute(int $args) : void {
        $num = $this->registers->read($args & 0b111);
        if($args & 0b100000 === 0b100000) {
            $this->registers->write($args & 0b111, $num >> 1);
        } else {
            $this->registers->write($args & 0b111, $num << 1);
        }
    }
}

class JMPC extends Opcode {
    private Components $components;
    private Registers $registers;
    private Register $programCounter;
    private int $depth;

    public function __construct(
        Components &$components,
        Registers &$registers,
        int &$depth,
        Register &$programCounter
    ) {
        $this->components = $components;
        $this->registers = $registers;
        $this->programCounter = $programCounter;
        $this->depth = $depth;
    }

    public function execute(int $args) : void {
        if($this->registers->read($args & 0b111) === 0) {
            $address = 0;
            for($offset = 1; $offset <= $this->depth; $offset++) {
                $address += $this->components->read($this->programCounter->read() + $offset) << (8 * ($this->depth - $offset));
            }
            if($address === $this->programCounter->read()) {
                die();
            }
            $this->programCounter->write($address - 1);
        } else {
            $this->programCounter->write($this->programCounter->read() + $this->depth);
        }
    }
}